/*
	
	MultiLevelLayoutPlugin for Cytoscape (http://www.cytoscape.org/) 
	Copyright (C) 2007 Pekka Salmela

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
	
 */

package multilevelLayoutPlugin;

import java.awt.event.ActionEvent;

import javax.swing.JOptionPane;

import cytoscape.CyNetwork;
import cytoscape.Cytoscape;
import cytoscape.data.CyAttributes;
import cytoscape.plugin.CytoscapePlugin;
import cytoscape.task.Task;
import cytoscape.task.TaskMonitor;
import cytoscape.task.ui.JTaskConfig;
import cytoscape.task.util.TaskManager;
import cytoscape.util.CytoscapeAction;
import cytoscape.view.CyNetworkView;

/**
 * The main class of this plugin. 
 * It is used to create a menu entry for launhing the plugin and 
 * creating a thread for running the layout calculation. 
 * 
 * @author Pekka Salmela
 *
 */
public class MultilevelLayoutPlugin extends CytoscapePlugin {

	/**
	 * Class constructor.
	 */
	public MultilevelLayoutPlugin() {
		//create a new action to respond to menu activation
        MultilevelLayoutSelectionAction action = new MultilevelLayoutSelectionAction();
        //set the preferred menu
        action.setPreferredMenu("Plugins");
        //and add it to the menus
        Cytoscape.getDesktop().getCyMenus().addAction(action);
	}

	/**
	 * Gets a description of this plugin.
	 * @return Description of this plugin.
	 */
    public String describe() {
        StringBuffer sb = new StringBuffer();
        sb.append("Calculates a nice layout for the network using multilevel");
        sb.append("force-directed algorithm by C. Walshaw.");
        return sb.toString();
    }
    
    /**
     * This class gets attached to the menu item.
     */
    public class MultilevelLayoutSelectionAction extends CytoscapeAction implements Task {
        
    	/**
    	 * Task monitor used to report calculation progress.
    	 */
    	private TaskMonitor taskMonitor;
    	/**
    	 * Instance of the class used for layout calculation. 
    	 */
    	private MultilevelLayout algObj;
    	/**
    	 * The graph the layout is calculated for. 
    	 */
    	private CyNetwork network;
    	/**
    	 * View of the graph the layout is calculated for.
    	 */
    	private CyNetworkView view;
    	
		private static final long serialVersionUID = -4840201619467047796L;

		/**
         * Class constructor.
         */
        public MultilevelLayoutSelectionAction(){
        	super("Multilevel Layout");
        }
        
        private boolean ready = false;
        
        /**
         * Launches the plugin when the appropriate menu item is 
         * selected by the user.
         * @param ae Event triggered when the menu item is selected.
         */
        public void actionPerformed(ActionEvent ae){
        	System.out.println("Start MultiLevelPlugin");
        	
        	//get the network object
            network = Cytoscape.getCurrentNetwork();
            //get the network view object
            view = Cytoscape.getCurrentNetworkView();
            //can't continue if either of these is null
            if (network == null || view == null) {
            	JOptionPane.showMessageDialog(Cytoscape.getDesktop(), "There was no graph or graph view available, layout " +
            			"calculation aborted.", "Error: No graph or graph view available", JOptionPane.ERROR_MESSAGE);
            	return;
            }
            if (network.getNodeCount() == 0) {
            	JOptionPane.showMessageDialog(Cytoscape.getDesktop(), "Current graph contains 0 nodes, " +
            			"calculation aborted.", "Error: Zero nodes in current graph", JOptionPane.ERROR_MESSAGE);
            	return;
            }
    		if(!GraphConnectivityChecker.graphIsConnected(network)){
    			JOptionPane.showMessageDialog( Cytoscape.getDesktop(), "The graph is not connected. This version of " +
    					"MultiLevelLayout plugin only supports connected graphs.");
    			return;
    		}
            //apply the MultilevelLayout
            ready = true;
            JTaskConfig taskConfig = getNewDefaultTaskConfig();
            TaskManager.executeTask(this, taskConfig);
            
    		CyAttributes nodesAttributes = Cytoscape.getNodeAttributes();
    		nodesAttributes.deleteAttribute("ml_previous");
    		nodesAttributes.deleteAttribute("ml_ancestor1");
    		nodesAttributes.deleteAttribute("ml_ancestor2");
    		nodesAttributes.deleteAttribute("ml_weight");
    		nodesAttributes.deleteAttribute("mllp_partition");
            
            System.out.println("Stop MultiLevelPlugin");
        }
        
    	/**
    	 * Runs the thread used for the actual layout calculation.
    	 */
        public void run(){
        	if(ready){
        		algObj = new MultilevelLayout(view);
        		algObj.setTaskMonitor(taskMonitor);
        		algObj.construct();
        		algObj = null;
        		System.gc();
        	}
    	}
        
        /**
         * Sets the task monitor to be used.
         */
    	public void setTaskMonitor(TaskMonitor _monitor){
    		taskMonitor = _monitor;
    	}
    	
    	/**
    	 * Gets the title of this task.
    	 * @return Title of this task.
    	 */
    	public String getTitle(){ 
    		return "Performing MultiLevelLayout"; 
    	}
    	
    	/**
    	 * Informs the algorithm it should be aborted.
    	 */
    	public void halt(){ 
    		algObj.setCancel();
    	}
    	
    	/**
    	 * Creates a new task configuration.
    	 */
    	private JTaskConfig getNewDefaultTaskConfig()
    	{
    		JTaskConfig result = new JTaskConfig();

    		result.displayCancelButton(true);
    		result.displayCloseButton(true);
    		result.displayStatus(true);
    		result.displayTimeElapsed(true);
    		result.setAutoDispose(false);
    		result.setModal(true);
    		result.setOwner(Cytoscape.getDesktop());

    		return result;
    	}
    }
}
