/*
	
	MultiLevelLayoutPlugin for Cytoscape (http://www.cytoscape.org/) 
	Copyright (C) 2007 Pekka Salmela

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
	
 */

package multilevelLayoutPlugin;

import giny.model.Node;
import giny.view.NodeView;

import java.util.Iterator;
import java.util.Vector;

import csplugins.layout.AbstractLayout;
import cytoscape.CyEdge;
import cytoscape.CyNetwork;
import cytoscape.CyNode;
import cytoscape.Cytoscape;
import cytoscape.view.CyNetworkView;
import cytoscape.data.CyAttributes;
import cytoscape.task.TaskMonitor;

/**
 * Class executing the multilevel layout algorithm originally
 * presented by C. Walshaw (2003). The algorithm is slightly 
 * tuned by Pekka Salmela (2007). The algorithm contains two
 * main phases:
 * 1) Construct a hierarchy of continuously coarser graphs by 
 * finding maximal independent sets of edges on each level until
 * there are only two nodes and one edge left.
 * 2) Starting from the coarsest graph consisting, place the two 
 * nodes randomly and use the positions to place nodes of the next 
 * graph in the hierarchy. Then adjust the positions using a 
 * heavily tuned version of force-directed placement algorithm.
 *    
 * @author Pekka Salmela
 */

public class MultilevelLayout extends AbstractLayout{
	
	/**
	 * Container used to store node positions. 
	 */
	private NodePositionManager posManager;
	
	/**
	 * The level on which the layout algorithm is currently working on.
	 */
	private double level;
	
	/**
	 * The task monitor that we report to
	 */
	protected TaskMonitor taskMonitor;
	
	/**
	 * A flag telling us if the user has requested a cancel.
	 */
	protected boolean cancel = false;

	public MultilevelLayout(CyNetworkView networkView){
		super(networkView);
	}
	  
	/**
	 * Main entry point for AbstractLayout classes. Initializes and
	 * runs the algorithm.
	 * @return Always returns <code>null</code>.
	 */
	public Object construct() {
		taskMonitor.setStatus("Initializing");
		initialize();  // Calls initialize_local
		layout();
		networkView.fitContent();
		networkView.updateView();
		return null;
	}
	  
	/**
	 * Sets the task monitor to be used.
	 * @param t Task monitor to be used.
	 */
	public void setTaskMonitor(TaskMonitor t) {
			this.taskMonitor = t;
	}
	
	/**
	 * Tells the algorithm it should be aborted.
	 *  
	 */
	public void setCancel() {
		this.cancel = true;
	}
		
	/**
	 * Call all of the initialization code.  Called from
	 * <code>AbstractLayout.initialize()</code>.
	 * 
	 */
	protected void initialize_local() {
		posManager = new NodePositionManager(networkView.getNetwork().getNodeCount());
		level = 0;
	}
		
	/**
	 * Execute layout algorithm for the graph given when 
	 * calling the constructor method.  
	 *
	 */
	@SuppressWarnings("unchecked")
	public void layout(){
		long start = System.currentTimeMillis();
		
		CyNetwork cn = networkView.getNetwork();

		if(cancel == true) return;
		
		taskMonitor.setStatus("Finding independent sets...");
		taskMonitor.setPercentCompleted(5);
		
		Vector<CyNetwork> networkSet = new Vector<CyNetwork>((int)Math.sqrt(cn.getNodeCount()), 5);
		networkSet.add(cn);
		boolean goOn = true;
		/*-------------FIRST LOOP-------------*/
		//construct a set of continuously coarser graphs by calculating maximum independent sets
		int nodeCount = cn.getNodeCount();
		while(goOn){
			CyNetwork next = MaximalIndependentSetFinder.findMaximalIndependentSet(networkSet.lastElement(), level);
			if(nodeCount != next.getNodeCount()){
				nodeCount = next.getNodeCount();
				if(nodeCount == 2) goOn = false;
				networkSet.add(next);
				level++;
			}
			else goOn = false;
		}
		/*-------------END FIRST LOOP-------------*/
		
		if(cancel == true) {
			for(int i = 1; i<networkSet.size(); i++){Cytoscape.destroyNetwork(networkSet.elementAt(i));}
			return;
		}

		taskMonitor.setStatus("Calculating the layout...");
		taskMonitor.setPercentCompleted(10);
		
		CyNetwork currentNetwork = null;
		double previousNaturalSpringLength = 0.0;

		/*-------------SECOND LOOP-------------*/
		//starting from the coarsest graph, calculate enhanced force-directed layout for each level
		while(networkSet.size()>1){
			if(cancel == true) {
				for(int i = 1; i<networkSet.size(); i++){Cytoscape.destroyNetwork(networkSet.elementAt(i));}
				return;
			}
			level--;
			taskMonitor.setStatus("Calculating the layout on level " + (int)level + "...");
			taskMonitor.setPercentCompleted((int)(80/(level+2) + 10));
			//if currentNetwork is not set, then this is the first level
			if(currentNetwork == null){
				currentNetwork = networkSet.lastElement();
				double max = 10.0/(Math.pow(Math.sqrt(4.0/7.0), level+1.0));
				Iterator<CyNode> iter = currentNetwork.nodesIterator();
				CyNode n = iter.next();
				posManager.addNode(n.getRootGraphIndex(), max, 0.0);
				n = iter.next();
				posManager.addNode(n.getRootGraphIndex(), 0.0, max);
				previousNaturalSpringLength = Math.sqrt(2.0) * max;
			}
			//take the network before the last one from the list
			CyNetwork nextNetwork = networkSet.elementAt(networkSet.size()-2);
			//use nextNetwork to place nodes in currentNetwork
			doOneLevelPlacement(currentNetwork, nextNetwork, previousNaturalSpringLength);
			//use EFDL to nextNetwork
			EnhancedForceDirectedLayout e = new EnhancedForceDirectedLayout(previousNaturalSpringLength, level, nextNetwork, posManager);
			e.doLayout();
			//store the natural spring length
			previousNaturalSpringLength = e.getK();
			//remove used network from the list
			networkSet.remove(networkSet.size()-1);
			
			//clean up (we can not use Cytoscape.destroyNetwork(currentNetwork), because
			//currentNetwork has been created by Cytoscape.getRootGraph.createNetwork(...)
			//and destroying raises an exception)
			Iterator<CyNode> i1 = currentNetwork.nodesIterator();
			while(i1.hasNext()){
				CyNode n = i1.next();
				posManager.removeNode(n.getRootGraphIndex());
				Cytoscape.getRootGraph().removeNode(n);
			}
			Iterator<CyEdge> i2 = currentNetwork.edgesIterator();
			while(i2.hasNext()){Cytoscape.getRootGraph().removeEdge(i2.next());}
			
			//change the network to be used during the next iteration
			currentNetwork = nextNetwork;
		}
		/*-------------END SECOND LOOP-------------*/
		
		//iterate over node positions and apply them to the NetworkView
		taskMonitor.setStatus("Updating node positions...");
		taskMonitor.setPercentCompleted(95);
		
		//scale the network to look good, or at least tolerable
		double minX = Double.POSITIVE_INFINITY;
		double minY = Double.POSITIVE_INFINITY;
		double maxX = Double.NEGATIVE_INFINITY;
		double maxY = Double.NEGATIVE_INFINITY;
		
		Iterator<Node> iter = currentNetwork.nodesIterator();
		while(iter.hasNext()){
			Node n = iter.next();
			//NodeView nv  = finalView.getNodeView(iter.next());
			if(posManager.getX(n.getRootGraphIndex()) < minX) minX = posManager.getX(n.getRootGraphIndex());
			if(posManager.getY(n.getRootGraphIndex()) < minY) minY = posManager.getY(n.getRootGraphIndex());
			if(posManager.getX(n.getRootGraphIndex()) > maxX) maxX = posManager.getX(n.getRootGraphIndex());
			if(posManager.getY(n.getRootGraphIndex()) > maxY) maxY = posManager.getY(n.getRootGraphIndex());
		}
		iter = currentNetwork.nodesIterator();
		while(iter.hasNext()){
			Node n = iter.next();
			posManager.setX(n.getRootGraphIndex(), ((300.0 * Math.sqrt((double)currentNetwork.getNodeCount()) * (posManager.getX(n.getRootGraphIndex())-minX))/(maxX-minX)));
			posManager.setY(n.getRootGraphIndex(), ((300.0 * Math.sqrt((double)currentNetwork.getNodeCount()) * (posManager.getY(n.getRootGraphIndex())-minY))/(maxY-minY)));
		}
		
		taskMonitor.setStatus("Laying out the graph...");
		taskMonitor.setPercentCompleted(99);
		
		CyNetworkView finalView = Cytoscape.getCurrentNetworkView();
		iter = currentNetwork.nodesIterator();
		while(iter.hasNext()){
			Node n = iter.next();
			NodeView nv  = finalView.getNodeView(n);
			nv.setXPosition(posManager.getX(n.getRootGraphIndex()));
			nv.setYPosition(posManager.getY(n.getRootGraphIndex()));
		}
		networkSet = null;
		posManager = null;
		
		System.out.println("Calculating the layout took " + ((System.currentTimeMillis() -start)/1000.0) + " seconds.");
		
		taskMonitor.setStatus("Layout complete");
		taskMonitor.setPercentCompleted(100);
		
	}
	
	/**
	 * Places the nodes of a more detailed graph using the positions of
	 * a coarser graph. A position manager is used to store node 
	 * positions and node attributes are used to denote connections between
	 * nodes in different graph levels.
	 * @param coarser The graph determining the node positions.
	 * @param finer The graph the placement is applied to. 
	 */
	@SuppressWarnings("unchecked")
	private void doOneLevelPlacement (CyNetwork coarser, CyNetwork finer, double k){
		//iterate over the nodes of the previous graph
		Iterator<CyNode> nodesIterator = coarser.nodesIterator();
		CyAttributes nodesAttributes = Cytoscape.getNodeAttributes();

		while(nodesIterator.hasNext()){
			CyNode n = nodesIterator.next();
			double nX = posManager.getX(n.getRootGraphIndex());
			double nY = posManager.getY(n.getRootGraphIndex());
			if(nodesAttributes.getIntegerAttribute(n.getIdentifier(), "ml_previous") != null){
				//place only one node
				Integer pre = nodesAttributes.getIntegerAttribute(n.getIdentifier(), "ml_previous");
				posManager.addNode(pre, nX, nY);
			}
			else{
				//place two nodes
				Integer anc1 = nodesAttributes.getIntegerAttribute(n.getIdentifier(), "ml_ancestor1");
				Integer anc2 = nodesAttributes.getIntegerAttribute(n.getIdentifier(), "ml_ancestor2");
				posManager.addNode(anc1, nX, nY);
				posManager.addNode(anc2, nX + plusOrMinusOne()*0.001*k, nY + plusOrMinusOne()*0.001*k);
			}
		}
	}
	
	/**
	 * Returns 1 or -1 with equal probability.
	 * @return 1 or -1, toss a coin to guess which one.
	 */
	private double plusOrMinusOne(){
		if(Math.random()<0.5) return 1.0;
		else return -1.0;
	}
}
